% MAIN.m

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DESCRIPTION:
%
% This program implements the Phelan/Skrzypacz equilibrium checking 
% algorithm.  It can handle 2-player games with an arbitrary number of 
% actions, signals, and states. 
%
% HOW TO USE IT:
%
% Step 1. Go to the "Set Inputs" section below and input your parameters
%         of interest.  (Do not modify other parts of the code unless you
%         know what you're doing.)
%
% Step 2. Run the program from Matlab.  The results will be displayed in
%         the command window and exported to results.txt.
%
% NOTATION:
% 
% plr(i).u(ai,yi)         = u_i(ai,yi)
% plr(i).p(ai,wi)         = p_i(ai|wi)
% plr(i).wp(wi,ai,yi)     = w_i+(wi,ai,yi)
% plr(i).w0(si)           = w_{0i}(si)
%
% plr(i).v(wi,wni)        = v_i(wi,wni)
% plr(i).v_dev            = i's continuation utility from deviating
%                           to ai in state wi when the opponent is
%                           in state wni
% plr(i).M(wi).emp        = 1 if Mi(wi) is empty; =0 otherwise
% plr(i).M(wi).ext(j,wni) = mi(wni), where mi is the j-th extreme
%                           point of Mi(wi)
% plr(i).ic(wi).dd(ai,j)  = 1 if i has the desire to deviate to ai
%                           in state wi when his belief is the j-th 
%                           extreme point of Mi(wi); 0 otherwise
% 
% Author:
%
% Kenichi Fukushima
% University of Minnesota and FRB-Minneapolis
% fuku0028@umn.edu
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%
% Initialization %
%%%%%%%%%%%%%%%%%%

% clear workspace and previous results

clear
close all

delete results.txt;

% start recording results

diary results.txt;

% declare global

global beta plr Pr_data x_data n NA NY NW jstate

% named constants

n = [2 1]; % n for "not"
EMP = -99.0; % named constant indicating emptiness 

% define player construct

plr = struct;

%%%%%%%%%%%%%%
% Set Inputs %
%%%%%%%%%%%%%%

%-------------------
% config parameters
%-------------------

% number of actions

NA = [2,2]; % NA(i)=|A_i|

% number of signals

NY = [2,2]; % NY(i)=|Y_i|

% number of states

NW = [2,2]; % NW(i)=|\Omega_i|

%------------------------------------------------
% allocating memory ... do not modify this part!
%------------------------------------------------

% shared objects

Pr_data = zeros(NY(1),NY(2),NA(1),NA(2));
x_data = zeros(NW(1),NW(2));

% player specific objects

for i=1:2
  
  % inputs
  
  plr(i).u = zeros(NA(i),NY(i));    
  plr(i).p = zeros(NA(i),NW(i));
  plr(i).wp = zeros(NW(i),NA(i),NY(i));
  plr(i).w0 = zeros(1,NW(i));
  
  % solution objects
  
  plr(i).v = zeros(NW(i),NW(n(i)));
  plr(i).v_dev = zeros(NW(i),NW(n(i)),NA(i));
  plr(i).M(NW(i)) = struct('ext',zeros(NW(n(i)),NW(n(i))),'emp',0);
  plr(i).ic(NW(i)) = struct('dd',zeros(NA(i),NW(n(i))));

end

% construct jstate (joint state converter)

jstate = zeros(NW(1)*NW(2),2);

w = 1;

for w1=1:NW(1); for w2=1:NW(2)
    jstate(w,:) = [w1 w2];
    w = w+1; 
end; end

%------------------
% model parameters
%------------------

% discount factor: beta

beta = 0.90;

% period payoffs: plr(i).u(ai,yi)=ui(ai,yi)

plr(1).u(1,1) = 0.6;
plr(1).u(1,2) = -0.4;
plr(1).u(2,1) = 1.0;
plr(1).u(2,2) = 0.0;

plr(2).u(1,1) = 0.6;
plr(2).u(1,2) = -0.4;
plr(2).u(2,1) = 1.0;
plr(2).u(2,2) = 0.0;

% signal probabilities: Pr_data(y1,y2,a1,a2)=P(y1,y2|a1,a2)

Pr_data(1,1,1,1) = 0.8556;
Pr_data(1,2,1,1) = 0.0244;
Pr_data(2,1,1,1) = 0.0244;
Pr_data(2,2,1,1) = 0.0956;

Pr_data(1,1,1,2) = 0.5231;
Pr_data(1,2,1,2) = 0.0244;
Pr_data(2,1,1,2) = 0.0244;
Pr_data(2,2,1,2) = 0.4281;

Pr_data(1,1,2,1) = 0.5231;
Pr_data(1,2,2,1) = 0.0244;
Pr_data(2,1,2,1) = 0.0244;
Pr_data(2,2,2,1) = 0.4281;

Pr_data(1,1,2,2) = 0.2856;
Pr_data(1,2,2,2) = 0.0244;
Pr_data(2,1,2,2) = 0.0244;
Pr_data(2,2,2,2) = 0.6656;

% correlation device (all entries must be >0): x_data(s1,s2)=x(s1,s2)

x_data(1,1) = 0.25;
x_data(1,2) = 0.25;
x_data(2,1) = 0.25;
x_data(2,2) = 0.25;

%---------------------
% strategy parameters
%---------------------

% action probabilities: plr(i).p(ai,wi)=pi(ai|wi)

plr(1).p(1,1) = 1.0;
plr(1).p(2,1) = 0.0;
plr(1).p(1,2) = 0.0;
plr(1).p(2,2) = 1.0;

plr(2).p(1,1) = 1.0;
plr(2).p(2,1) = 0.0;
plr(2).p(1,2) = 0.0;
plr(2).p(2,2) = 1.0;

% state transition rules: plr(i).wp(wi,ai,yi)=wi+(wi,ai,yi)

plr(1).wp(1,1,1) = 1;
plr(1).wp(1,1,2) = 2;
plr(1).wp(1,2,1) = 1;
plr(1).wp(1,2,2) = 2;
plr(1).wp(2,1,1) = 1;
plr(1).wp(2,1,2) = 2;
plr(1).wp(2,2,1) = 1;
plr(1).wp(2,2,2) = 2;

plr(2).wp(1,1,1) = 1;
plr(2).wp(1,1,2) = 2;
plr(2).wp(1,2,1) = 1;
plr(2).wp(1,2,2) = 2;
plr(2).wp(2,1,1) = 1;
plr(2).wp(2,1,2) = 2;
plr(2).wp(2,2,1) = 1;
plr(2).wp(2,2,2) = 2;

% initial states: plr(i).w0(si)=w_{0,i}(si)

plr(1).w0(1) = 1;
plr(1).w0(2) = 2;

plr(2).w0(1) = 1;
plr(2).w0(2) = 2;

%%%%%%%%%%%%%%%
% Echo inputs %
%%%%%%%%%%%%%%%

echo_inputs();

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute cont payoff vi and cont deviation payoff vi_dev %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for i=1:2
  [plr(i).v,plr(i).v_dev] = get_v(i);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute Mbar and evaluate incentives on its basis %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% computation

for i=1:2
    
    % iterate on T to get Mbar_i
    
    plr(i).M = lim_T(i,get_Mmax(i));
        
    % check incentives
    
    plr(i).ic = check_ic(i);
        
end

% report results

disp('========================================================')
disp('Part I: Is the strategy''s behavior consistent with CSE?')
disp('        -- Evaluation based on Mbar')
disp('========================================================')

report_results('Mbar')

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute the invariant distributions pi and the implied M^*_{\pi} %
% Then evaluate incentives on their basis                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% get the invariant distributions and report

[pis,npi] = get_pis();

disp('=========================================================')
disp('Part II: Is the strategy''s behavior consistent with CSE?')
disp('         -- Evaluation based on M_pi^*')
disp('=========================================================')
disp(' ')
disp(['There are ', num2str(npi), ' stationary state distributions.'])
disp('Their values and associated results are the following:')
disp(' ')

% compute and report results for each pi

for k=1:npi % loop over all invariant distributions

    % compute M_pi^* and check incentives
    
    for i=1:2
        
        % iterate on T to get M_pistar
        
        plr(i).M = lim_T(i,get_Mpi(i,pis(:,k)));
        
        % check incentives
        
        plr(i).ic = check_ic(i);
        
    end
    
    % report results
        
    disp('----------------------------------------')
    disp(['Results from stationary distribution # ', num2str(k)])
    disp('----------------------------------------')
    disp(' ')
    disp('0. The stationary distribution pi is:')
    disp(' ')
    
    for w=1:NW(1)*NW(2)
      
      w1_s = num2str(jstate(w,1));
      w2_s = num2str(jstate(w,2));
      
      disp([' pi(w1=',w1_s,',w2=',w2_s,') = ', num2str(pis(w,k))])
      
    end
    
    report_results('M_pi^*')
      
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute M^{U*} for the user-supplied starting condition and evaluate %
% incentives on their basis                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% computation

for i=1:2
        
  % iterate up using TU

  plr(i).M = lim_TU(i,get_Mi0(i));

  % check incentives

  plr(i).ic = check_ic(i);
        
end

% report results

disp('===============================================================')
disp('Part III: Is (x,sigma) a CSE?')
disp('          -- Evaluation using user-supplied starting conditions')
disp('===============================================================')

report_results('M^{U*}')

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute M^{U*} for each deterministic starting condition and evaluate %
% incentives on their basis                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% heading

disp('==================================================================')
disp('Part IV: Is (x,sigma) a CSE?')
disp('         -- Evaluation using all deterministic starting conditions')
disp('==================================================================')

% computation and reporting

x_data(:,:) = 0.25; % overwrite x_data

for w=1:NW(1)*NW(2) % loop over all deterministic starting conditions
    
    % set initial conditions and display header (overwrite w0)
    
    plr(1).w0(:) = jstate(w,1);
    plr(2).w0(:) = jstate(w,2);
    
    w01_s = num2str(jstate(w,1));
    w02_s = num2str(jstate(w,2));
    
    disp('------------------------------------------')
    disp(['Results for (w_{0,1},w_{0,2})=(',w01_s,',',w02_s,') w.p. 1'])
    disp('------------------------------------------')   
            
    % for each i, compute M_i^{U*} and check ic 
    
    for i=1:2
        
        % iterate up using TU
        
        plr(i).M = lim_TU(i,get_Mi0(i));
        
        % check incentives
        
        plr(i).ic = check_ic(i);
        
    end
    
    % report results
    
    report_results('M^{U*}')
    
end
  
% end
% tail of the file
diary off;
